import Product from '../models/Product.js';
import { fetchAliExpressProduct } from '../services/aliexpressService.js';

/**
 * Get all products from MongoDB
 */
export const getAllProducts = async (req, res) => {
  try {
    const products = await Product.find();
    res.json(products);
  } catch (error) {
    console.error('Error fetching products:', error);
    res.status(500).json({ error: 'Server error' });
  }
};

/**
 * Get a single product by ID
 */
export const getProductById = async (req, res) => {
  try {
    const product = await Product.findById(req.params.id);
    if (!product) return res.status(404).json({ error: 'Product not found' });
    res.json(product);
  } catch (error) {
    console.error('Error fetching product:', error);
    res.status(500).json({ error: 'Server error' });
  }
};

/**
 * Create a new product
 */
export const createProduct = async (req, res) => {
  try {
    const { name, price, image_url, category } = req.body;
    const product = await Product.create({ name, price, image_url, category });
    res.status(201).json(product);
  } catch (error) {
    console.error('Error creating product:', error);
    res.status(500).json({ error: 'Server error' });
  }
};

/**
 * Update an existing product
 */
export const updateProduct = async (req, res) => {
  try {
    const updated = await Product.findByIdAndUpdate(req.params.id, req.body, { new: true });
    if (!updated) return res.status(404).json({ error: 'Product not found' });
    res.json(updated);
  } catch (error) {
    console.error('Error updating product:', error);
    res.status(500).json({ error: 'Server error' });
  }
};

/**
 * Delete a product
 */
export const deleteProduct = async (req, res) => {
  try {
    const deleted = await Product.findByIdAndDelete(req.params.id);
    if (!deleted) return res.status(404).json({ error: 'Product not found' });
    res.json({ success: true });
  } catch (error) {
    console.error('Error deleting product:', error);
    res.status(500).json({ error: 'Server error' });
  }
};

/**
 * Import product from AliExpress
 */
export const importAliExpressProduct = async (req, res) => {
  try {
    const { url } = req.body;
    const productData = await fetchAliExpressProduct(url);

    const newProduct = new Product({
      name: productData.title || 'Unnamed Product',
      description: productData.description || 'No description available',
      price: productData.price?.value || 0,
      category: productData.category || 'Uncategorized',
      sourceUrl: url,
    });

    const savedProduct = await newProduct.save();
    res.status(201).json(savedProduct);
  } catch (error) {
    console.error('Error importing AliExpress product:', error);
    res.status(500).json({ error: 'Failed to import product' });

  }


  const Product = require('./models/Product');
const esClient = require('./esClient');

exports.fetchAliExpressProduct = async (url) => {
  try {
    // Simulate fetching product from RapidAPI or scraper
    const productData = {
      name: "Sample Product",
      price: 19.99,
      image_url: "https://example.com/image.jpg",
      category: "electronics",
      source_url: url
    };

    // Save to MongoDB
    const savedProduct = await Product.create(productData);

    // Index in Elasticsearch
    await esClient.index({
      index: 'products',
      id: savedProduct._id.toString(), // Use MongoDB ID as ES document ID
      document: {
        name: savedProduct.name,
        price: savedProduct.price,
        image_url: savedProduct.image_url,
        category: savedProduct.category,
        source_url: savedProduct.source_url
      }
    });

    return savedProduct;
  } catch (error) {
    console.error('Error fetching or saving product:', error);
    throw new Error('Failed to fetch and store product');
  }
};

};

