import bcrypt from 'bcryptjs';
import jwt from 'jsonwebtoken';
import User from '../models/userModel.js';
import { indexUser, searchUsers } from '../services/elasticService.js';

// Register a new user
export const registerUser = async (req, res) => {
  try {
    const { email, password, name } = req.body;
    const hash = await bcrypt.hash(password, 10);
    const user = await User.create({ name, email, password: hash });

    await indexUser(user); // Index user in Elasticsearch

    res.status(201).json({ id: user._id, email: user.email, name: user.name });
  } catch (error) {
    res.status(400).json({ message: error.message });
  }
};

// Login user and return JWT
export const loginUser = async (req, res) => {
  try {
    const { email, password } = req.body;
    const user = await User.findOne({ email });

    if (!user || !(await bcrypt.compare(password, user.password))) {
      return res.status(401).json({ message: 'Invalid credentials' });
    }

    const token = jwt.sign({ id: user._id }, process.env.JWT_SECRET, {
      expiresIn: '7d',
    });

    res.json({ token });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

// Search users via Elasticsearch
export const searchUser = async (req, res) => {
  try {
    const results = await searchUsers(req.query.q);
    res.json(results);
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

// Get current user's profile
export const getUserProfile = async (req, res) => {
  try {
    const user = await User.findById(req.user.id).select('-password');
    if (!user) return res.status(404).json({ message: 'User not found' });
    res.json(user);
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

// Update current user's profile
export const updateUserProfile = async (req, res) => {
  try {
    const user = await User.findById(req.user.id);
    if (!user) return res.status(404).json({ message: 'User not found' });

    user.name = req.body.name || user.name;
    user.email = req.body.email || user.email;

    const updatedUser = await user.save();
    await indexUser(updatedUser); // Re-index updated user

    res.json(updatedUser);
  } catch (error) {
    res.status(400).json({ message: error.message });
  }
};
