import express from 'express';
import cors from 'cors';
import mongoose from 'mongoose';
import dotenv from 'dotenv';

import config from './config/config.js';
import  db  from './db/connections.js'; // Use only one connectDB import
import  authMiddleware from './Middleware/authMiddleware.js';
import { notFound, errorHandler } from './Middleware/errorMiddleware.js';
import { otpLimiter } from './Middleware/rateLimiter.js';

import adminRoutes from './routes/adminRoutes.js';
import productRoutes from './routes/productRoutes.js';

import fetchAliExpressData from './utils/fetchAliExpressData.js';
import UserModel from './models/userModel.js';




dotenv.config(); // Load env variables early

const app = express();

// Middleware
app.use(cors());
app.use(express.json());

// Routes
app.use('/api/admin', adminRoutes);
app.use('/api/products', productRoutes);

// Admin product creation (placeholder)
app.post('/api/admin/products', auth, adminOnly, (req, res) => {
  res.status(501).json({ message: 'Not implemented' });
});

// OTP route with rate limiter
app.post('/api/auth/otp', otpLimiter, (req, res) => {
  res.status(501).json({ message: 'OTP route not implemented' });
});

// Example user creation route
app.post('/users', async (req, res) => {
  try {
    const user = new UserModel(req.body);
    await user.save();
    res.status(201).json(user);
  } catch (error) {
    res.status(400).json({ message: error.message });
  }
});

// Error handling
app.use(notFound);
app.use(errorHandler);

// Start server inside async IIFE
(async () => {
  try {
    await connectDB();

    const users = await Utils.query(UserModel, { active: true });
    console.log('Active users:', users);

    const PORT = config.port || 3000;
    app.listen(PORT, () => {
      console.log(`Backend running on port ${PORT}`);
    });
  } catch (error) {
    console.error('Failed to start server:', error);
  }
})();

import { connectElasticsearch, connectMongoDB } from './db/connections.js';

const startApp = async () => {
  const esClient = await connectElasticsearch();
  const mongoClient = await connectMongoDB();

  // You can now use esClient and mongoClient.db('your-db-name')...
};

startApp();
