import jwt from 'jsonwebtoken';
import User from '../models/userModel.js';

/**
 * Generic authentication middleware.
 * Verifies JWT and attaches user to request.
 */
const authMiddleware = async (req, res, next) => {
  const authHeader = req.headers.authorization;

  if (!authHeader || !authHeader.startsWith('Bearer ')) {
    return res.status(401).json({ message: 'Authorization token missing' });
  }

  const token = authHeader.split(' ')[1];

  try {
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    const user = await User.findById(decoded.id).select('-password');

    if (!user) {
      return res.status(401).json({ message: 'User not found' });
    }

    req.user = user;
    next();
  } catch (err) {
    return res.status(401).json({ message: 'Invalid or expired token' });
  }
};

/**
 * Protect route for authenticated users only.
 */
export const protectUser = authMiddleware;

/**
 * Protect route for admin users only.
 */
export const protectAdmin = async (req, res, next) => {
  try {
    await authMiddleware(req, res, async () => {
      if (!req.user?.isAdmin) {
        return res.status(403).json({ message: 'Admin access required' });
      }
      next();
    });
  } catch (err) {
    return res.status(403).json({ message: 'Admin access denied' });
  }
};

/**
 * Middleware to restrict access to admin users (requires user already authenticated).
 */
export const adminOnly = (req, res, next) => {
  if (!req.user?.isAdmin) {
    return res.status(403).json({ message: 'Admin access required' });
  }
  next();
};

/**
 * Optional: Static token-based admin authentication.
 */
export const staticAdminAuth = (req, res, next) => {
  const token = req.headers.authorization;
  if (!token || token !== `Bearer ${process.env.ADMIN_TOKEN}`) {
    return res.status(401).json({ error: 'Unauthorized' });
  }
  next();
};
export default {
  protectUser,
  protectAdmin,
  adminOnly,
  staticAdminAuth,
  authMiddleware
  
};