import express from 'express';
import { Parser } from 'json2csv';
import authMiddleware from '../Middleware/authMiddleware.js';
import Order from '../models/orderModel.js';
import Product from '../models/productModel.js';
import esClient from '../config/elasticsearch.js'; // ✅ Correct default import
import { MongoClient } from 'mongodb';
import config from '../config/config.js';

const router = express.Router();

// 🟡 Optional: Check Elasticsearch connection
const checkElasticsearch = async () => {
  try {
    const health = await esClient.cluster.health();
    console.log(`✅ Elasticsearch Connected: ${health.body.status}`);
  } catch (error) {
    console.error(`❌ Elasticsearch Connection Error: ${error.message}`);
    process.exit(1);
  }
};

// 🔵 MongoDB connection
const connectMongoDB = async () => {
  const mongoClient = new MongoClient(config.mongoUrl);
  try {
    await mongoClient.connect();
    console.log('✅ MongoDB Connected');
  } catch (error) {
    console.error(`❌ MongoDB Connection Error: ${error.message}`);
    process.exit(1);
  }
  return mongoClient;
};

// ✅ GET /api/admin/orders
router.get('/admin/orders', authMiddleware, async (req, res) => {
  try {
    const orders = await Order.find()
      .sort({ createdAt: -1 })
      .populate({ path: 'product_id', select: 'name image price' })
      .lean();

    res.status(200).json(orders);
  } catch (err) {
    console.error('[MongoDB] Error fetching orders:', err);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// ✅ GET /api/admin/orders/export
router.get('/admin/orders/export', authMiddleware, async (req, res) => {
  try {
    const orders = await Order.find()
      .sort({ createdAt: -1 })
      .populate({ path: 'product_id', select: 'name price' })
      .lean();

    const formatted = orders.map(order => ({
      ID: order._id,
      Email: order.customer_email || 'N/A',
      Quantity: order.quantity || 0,
      Date: order.createdAt?.toISOString() || '',
      Product: order.product_id?.name || 'N/A',
      Price: order.product_id?.price || 0
    }));

    const parser = new Parser({ header: true });
    const csv = parser.parse(formatted);

    res.header('Content-Type', 'text/csv');
    res.attachment('orders.csv');
    res.status(200).send(csv);
  } catch (err) {
    console.error('[CSV Export] Error exporting orders:', err);
    res.status(500).json({ error: 'Failed to export orders' });
  }
});

// ✅ GET /api/admin/orders/filter
router.get('/admin/orders/filter', authMiddleware, async (req, res) => {
  try {
    const { email, start_date, end_date, limit = 50, offset = 0 } = req.query;

    const mongoQuery = {};
    if (email) {
      mongoQuery.customer_email = { $regex: email, $options: 'i' };
    }

    if (start_date || end_date) {
      mongoQuery.createdAt = {};
      if (start_date) mongoQuery.createdAt.$gte = new Date(start_date);
      if (end_date) mongoQuery.createdAt.$lte = new Date(end_date);
    }

    let orders;

    if (typeof esClient?.search === 'function') {
      try {
        const esQuery = {
          index: 'orders',
          from: Number(offset),
          size: Number(limit),
          body: {
            query: {
              bool: {
                must: [
                  ...(email ? [{ match: { customer_email: email } }] : []),
                  ...(start_date || end_date
                    ? [{
                        range: {
                          createdAt: {
                            ...(start_date && { gte: start_date }),
                            ...(end_date && { lte: end_date })
                          }
                        }
                      }]
                    : [])
                ]
              }
            }
          }
        };

        const { hits } = await esClient.search(esQuery);
        const ids = hits.hits.map(hit => hit._id);

        orders = await Order.find({ _id: { $in: ids } })
          .populate({ path: 'product_id', select: 'name image price' })
          .lean();
      } catch (esErr) {
        console.warn('[Elasticsearch] Fallback to MongoDB:', esErr.message);
        orders = await Order.find(mongoQuery)
          .sort({ createdAt: -1 })
          .skip(Number(offset))
          .limit(Number(limit))
          .populate({ path: 'product_id', select: 'name image price' })
          .lean();
      }
    } else {
      orders = await Order.find(mongoQuery)
        .sort({ createdAt: -1 })
        .skip(Number(offset))
        .limit(Number(limit))
        .populate({ path: 'product_id', select: 'name image price' })
        .lean();
    }

    res.status(200).json(orders);
  } catch (err) {
    console.error('[Filter Orders] Error:', err);
    res.status(500).json({ error: 'Failed to fetch filtered orders' });
  }
});

// ✅ PUT /api/admin/orders/:id - Update and reindex order
router.put('/admin/orders/:id', authMiddleware, async (req, res) => {
  try {
    const { id } = req.params;
    const updates = req.body;

    const updatedOrder = await Order.findByIdAndUpdate(id, updates, { new: true }).lean();

    if (!updatedOrder) {
      return res.status(404).json({ error: 'Order not found' });
    }

    // Reindex in Elasticsearch
    await esClient.index({
      index: 'orders',
      id: updatedOrder._id.toString(),
      body: {
        customer_email: updatedOrder.customer_email,
        quantity: updatedOrder.quantity,
        product_id: updatedOrder.product_id,
        createdAt: updatedOrder.createdAt
      }
    });

    res.status(200).json(updatedOrder);
  } catch (err) {
    console.error('[Update Order] Error:', err);
    res.status(500).json({ error: 'Failed to update order' });
  }
});

// ✅ DELETE /api/admin/orders/:id - Delete and remove from ES
router.delete('/admin/orders/:id', authMiddleware, async (req, res) => {
  try {
    const { id } = req.params;

    const deletedOrder = await Order.findByIdAndDelete(id);

    if (!deletedOrder) {
      return res.status(404).json({ error: 'Order not found' });
    }

    // Remove from Elasticsearch
    await esClient.delete({
      index: 'orders',
      id: id
    });

    res.status(200).json({ message: 'Order deleted successfully' });
  } catch (err) {
    console.error('[Delete Order] Error:', err);
    res.status(500).json({ error: 'Failed to delete order' });
  }
});



const syncOrdersToElasticsearch = async () => {
  try {
    const orders = await Order.find().lean();

    const bulkOps = orders.flatMap(order => [
      { index: { _index: 'orders', _id: order._id.toString() } },
      {
        customer_email: order.customer_email,
        quantity: order.quantity,
        product_id: order.product_id,
        createdAt: order.createdAt
      }
    ]);

    const { body } = await esClient.bulk({ refresh: true, body: bulkOps });

    if (body.errors) {
      console.warn('[Sync] Some documents failed to index');
    } else {
      console.log(`✅ Synced ${orders.length} orders to Elasticsearch`);
    }
  } catch (err) {
    console.error('[Sync Orders] Error:', err);
  }
};


// 🧩 Optional exports for diagnostics
export { checkElasticsearch, connectMongoDB, syncOrdersToElasticsearch };
export default router;
