
const express = require("express");
const axios = require("axios");
const { Parser } = require('json2csv');
const db = require("../config/config.js");
const adminController = require("../controllers/adminController.js");
const { protectAdmin } = require("../Middleware/authMiddleware.js");
const auth = require("../Middleware/auth.js");
const multer = require('multer');
const upload = multer({ dest: 'uploads/' });

const router = express.Router();

// Admin user, order, and product routes
router.get("/users", protectAdmin, adminController.getAllUsers);
router.get("/orders", protectAdmin, adminController.getAllOrders);
router.get("/products", protectAdmin, adminController.getAllProducts);

// AliExpress product import (controller-based)
router.post("/products/import", protectAdmin, adminController.importAliExpressProduct);

// Export orders as CSV (controller-based)
router.get("/orders/export", protectAdmin, adminController.exportOrdersCSV);

// Example: update order status
router.put('/orders/:id', protectAdmin, async (req, res) => {
  const { status, tracking_number } = req.body;
  await db.query(
    'UPDATE orders SET status = $1, tracking_number = $2 WHERE id = $3',
    [status, tracking_number, req.params.id]
  );
  res.json({ success: true });
});

// AliExpress product import (direct implementation)
router.post('/import', protectAdmin, async (req, res) => {
  const { query = 'electronics' } = req.body;

  try {
    const aliRes = await axios.get('https://ali-express1.p.rapidapi.com/search', {
      params: { query },
      headers: {
        'X-RapidAPI-Key': process.env.ALIEXPRESS_API_KEY,
        'X-RapidAPI-Host': 'ali-express1.p.rapidapi.com',
      },
    });

    const products = aliRes.data.docs.map(item => ({
      id: item.product_id,
      name: item.product_title,
      image: item.product_main_image_url,
      price: parseFloat(item.app_sale_price),
      category: query,
    }));

    for (let p of products) {
      await db.query(
        'INSERT INTO products (id, name, image, price, category) VALUES ($1, $2, $3, $4, $5) ON CONFLICT (id) DO NOTHING',
        [p.id, p.name, p.image, p.price, p.category]
      );
    }

    res.json({ success: true, inserted: products.length });
  } catch (err) {
    console.error(err);
    res.status(500).json({ error: 'Failed to import products' });
  }
});

// Export orders as CSV (direct implementation)
router.get('/orders/export/csv', protectAdmin, async (req, res) => {
  try {
    const result = await db.query(
      `SELECT o.id, o.customer_email, o.quantity, o.created_at, p.name AS product_name, p.price
       FROM orders o JOIN products p ON o.product_id = p.id
       ORDER BY o.created_at DESC`
    );

    const parser = new Parser();
    const csv = parser.parse(result.rows);

    res.header('Content-Type', 'text/csv');
    res.attachment('orders.csv');
    res.send(csv);
  } catch (err) {
    console.error(err);
    res.status(500).json({ error: 'Failed to export orders' });
  }
});

// Admin product importer (mock for now)
router.post('/admin/import', protectAdmin, async (req, res) => {
  const { query } = req.body;

  // Example: insert mock products
  const products = [
    { name: `${query} 1`, price: 10, image: 'https://via.placeholder.com/150' },
    { name: `${query} 2`, price: 15, image: 'https://via.placeholder.com/150' },
  ];

  const values = products.map(p => `('${p.name}', ${p.price}, '${p.image}')`).join(',');
  await db.query(`INSERT INTO products (name, price, image) VALUES ${values}`);

  res.json({ inserted: products.length });
});

// Sales analytics route
router.get('/sales-analytics', protectAdmin, async (req, res) => {
  try {
    const result = await db.query(`
      SELECT DATE(created_at) as date, COUNT(*) as orders, SUM(quantity * price) as revenue
      FROM orders o JOIN products p ON o.product_id = p.id
      GROUP BY date
      ORDER BY date ASC
    `);
    res.json(result.rows);
  } catch (err) {
    console.error(err);
    res.status(500).json({ error: 'Failed to fetch analytics' });
  }
});

export default router;


