import express from 'express';
import dotenv from 'dotenv';
import mongoose from 'mongoose';
import { Client } from '@elastic/elasticsearch';
import adminRoutes from './routes/adminRoutes.js';
import authRoutes from './routes/authRoutes.js';
import ordersRoutes from './routes/ordersRoutes.js';
import adminOrders from './routes/adminOrders.js';
import aliexpressRoutes from './routes/aliexpressRoutes.js';
import { connectDB } from './config/db.js';

connectDB();

dotenv.config();

const app = express();
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

// MongoDB connection
const connectMongoDB = async () => {
  try {
    await mongoose.connect(process.env.MONGO_URI, {
      useNewUrlParser: true,
      useUnifiedTopology: true,
    });
    console.log('✅ Connected to MongoDB');
  } catch (err) {
    console.error('❌ MongoDB connection error:', err);
    process.exit(1);
  }
};

// Elasticsearch connection
const esClient = new Client({ node: process.env.ELASTICSEARCH_URI });

const connectElasticsearch = async () => {
  try {
    const health = await esClient.cluster.health();
    console.log('✅ Connected to Elasticsearch:', health.body.status);
  } catch (err) {
    console.error('❌ Elasticsearch connection error:', err);
    process.exit(1);
  }
};

// Routes
app.use('/api/admin', adminRoutes);
app.use('/api', authRoutes);
app.use('/api', ordersRoutes);
app.use('/api', adminOrders);
app.use('/api/aliexpress', aliexpressRoutes);

// Example MongoDB + Elasticsearch route
app.get('/api/products', async (req, res) => {
  try {
    const { category, sort } = req.query;

    // MongoDB query
    const query = {};
    if (category) query.category = category;

    let products = await mongoose.model('Product').find(query);

    // Optional sorting
    if (sort === 'price_asc') products.sort((a, b) => a.price - b.price);
    else if (sort === 'price_desc') products.sort((a, b) => b.price - a.price);

    res.json(products);
  } catch (err) {
    console.error(err);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Error handling
app.use((err, req, res, next) => {
  console.error(err.stack);
  res.status(500).send('Something broke!');
});

// 404 handler
app.use((req, res) => {
  res.status(404).send('Not Found');
});

// Start server
const startServer = async () => {
  await connectMongoDB();
  await connectElasticsearch();

  const PORT = process.env.PORT || 3000;
  app.listen(PORT, () => console.log(`🚀 Server running on port ${PORT}`));
};


import index from './index.js';
dotenv.config();


connectDB(); // Establish MongoDB connection

console.log(`Server running on port ${config.port}`);

export const PORT = process.env.PORT || 5000;
export const NODE_ENV = process.env.NODE_ENV || 'development';

export const JWT_SECRET = process.env.JWT_SECRET || 'changeme';
export const JWT_EXPIRY = '7d';

export const DATABASE_URL = process.env.DATABASE_URL;
export const PAYPAL_CLIENT_ID = process.env.PAYPAL_CLIENT_ID;
export const PAYPAL_SECRET = process.env.PAYPAL_SECRET;



startServer();
